<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        // 1. Total Income (All collected payments)
        $totalIncome = \App\Models\Payment::sum('amount'); // You might want to filter by status if cheques bounce

        // 2. Pending Cash (Cash held by Reps that needs remitting)
        $pendingCash = \App\Models\Payment::where('method', 'Cash')
            ->where('is_remitted', false)
            ->sum('amount');
        
        // Subtract pending approved expenses from pending cash?
        // This is a quick approximation. For exact "Handover" amount per rep, we need grouped sums.
        // For dashboard global stat:
        $pendingExpenses = \App\Models\Expense::where('status', 'approved')
            ->where('is_reimbursed', false) // assuming we reimburse from cash collections
            ->sum('amount');
        
        $netPendingCash = max(0, $pendingCash - $pendingExpenses);

        // 3. Cheques (Pending Realization)
        $pendingCheques = \App\Models\Payment::where('method', 'Cheque')
            ->where('cheque_status', 'pending')
            ->sum('amount');

        // 4. Credit Sales (Total Outstanding Balance Due from Customers)
        $totalCredit = \App\Models\Sale::sum('balance_due');

        // --- Charts Data ---

        // 1. Weekly Sales Trend (Last 7 Days)
        $weeklyDates = [];
        $weeklySales = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $weeklyDates[] = $date->format('M d');
            $weeklySales[] = \App\Models\Sale::whereDate('created_at', $date->format('Y-m-d'))->sum('net_total');
        }

        // 2. Top 5 Products by Revenue
        $topItems = \App\Models\SaleItem::select('product_id', \Illuminate\Support\Facades\DB::raw('sum(total_price) as revenue'))
            ->groupBy('product_id')
            ->orderByDesc('revenue')
            ->take(5)
            ->with(['product' => function($q) { $q->withTrashed(); }]) // handle soft deleted products
            ->get();
        
        $topProductsLabels = $topItems->map(fn($item) => $item->product->name ?? 'Unknown')->toArray();
        $topProductsData = $topItems->pluck('revenue')->toArray();

        // 3. Payment Methods (Full History)
        $cashTotal = \App\Models\Payment::where('method', 'Cash')->sum('amount');
        $chequeTotal = \App\Models\Payment::where('method', 'Cheque')->sum('amount');

        return view('admin.dashboard', compact(
            'totalIncome', 'netPendingCash', 'pendingCheques', 'totalCredit',
            'weeklyDates', 'weeklySales',
            'topProductsLabels', 'topProductsData',
            'cashTotal', 'chequeTotal'
        ));
    }
}
