<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Delivery;
use App\Models\DeliveryItem;
use App\Models\Product;
use App\Models\ProductPackSize;
use App\Models\Stock;
use App\Models\User;
use App\Models\Vehicle;
use App\Models\VehicleStock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class DeliveryController extends Controller
{
    public function index(Request $request)
    {
        $query = Delivery::with(['vehicle', 'driver', 'salesRep']);

        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $deliveries = $query->latest()->paginate(10);
        return view('admin.deliveries.index', compact('deliveries'));
    }

    public function create()
    {
        $vehicles = Vehicle::where('is_active', true)->get();
        // Drivers: Role 'Driver' and NOT in an active delivery
        $drivers = User::whereHas('role', function($q) {
            $q->where('name', 'Driver');
        })->where('is_active', true)
        ->whereDoesntHave('driverDeliveries', function($q) {
            $q->whereIn('status', ['pending', 'loading', 'ongoing', 'unloading']);
        })->get();

        // Sales Reps: Role 'Sales Representative' and NOT in an active delivery
        $salesReps = User::whereHas('role', function($q) {
            $q->where('name', 'Sales Representative');
        })->where('is_active', true)
        ->whereDoesntHave('repDeliveries', function($q) {
            $q->whereIn('status', ['pending', 'loading', 'ongoing', 'unloading']);
        })->get();

        $products = Product::where('is_active', true)
        ->whereHas('stock', function($q) {
            $q->where('quantity', '>', 0);
        })
        ->with('packSizes.priceLevels')->get();

        return view('admin.deliveries.create', compact('vehicles', 'drivers', 'salesReps', 'products'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'driver_id' => 'required|exists:users,id',
            'sales_rep_id' => 'required|exists:users,id',
            'items' => 'required|array',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.product_pack_size_id' => 'required|exists:product_pack_sizes,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        try {
            DB::transaction(function () use ($request) {
                // 1. Create Delivery
                $delivery = Delivery::create([
                    'vehicle_id' => $request->vehicle_id,
                    'driver_id' => $request->driver_id,
                    'sales_rep_id' => $request->sales_rep_id,
                    'status' => 'loading',
                ]);

                // 2. Assign Vehicle to Sales Rep (for existing billing logic compatibility)
                $rep = User::find($request->sales_rep_id);
                $rep->vehicle_id = $request->vehicle_id;
                $rep->save();

                // 3. Process Items
                foreach ($request->items as $item) {
                    $packSize = ProductPackSize::find($item['product_pack_size_id']);
                    $totalBaseUnits = $item['quantity'] * $packSize->quantity;

                    // Check Warehouse Stock
                    $warehouseStock = Stock::where('product_id', $item['product_id'])->first();
                    if (!$warehouseStock || $warehouseStock->quantity < $totalBaseUnits) {
                        throw new \Exception("Insufficient warehouse stock for product: " . Product::find($item['product_id'])->name);
                    }

                    // Deduct from Warehouse
                    $warehouseStock->decrement('quantity', $totalBaseUnits);

                    // Add to Vehicle Stock
                    $vehicleStock = VehicleStock::firstOrCreate(
                        ['vehicle_id' => $request->vehicle_id, 'product_id' => $item['product_id']],
                        ['quantity' => 0]
                    );
                    $vehicleStock->increment('quantity', $totalBaseUnits);

                    // Create Delivery Item record
                    DeliveryItem::create([
                        'delivery_id' => $delivery->id,
                        'product_id' => $item['product_id'],
                        'loaded_quantity' => $totalBaseUnits,
                    ]);
                }
            });
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage())->withInput();
        }

        return redirect()->route('admin.deliveries.index')->with('success', 'Delivery created and truck loaded!');
    }

    public function start(Delivery $delivery)
    {
        if ($delivery->status !== 'loading') {
            return back()->with('error', 'Only loading deliveries can be started.');
        }

        $delivery->update([
            'status' => 'ongoing',
            'started_at' => now(),
        ]);

        return back()->with('success', 'Delivery started!');
    }

    public function end(Delivery $delivery)
    {
        if ($delivery->status !== 'ongoing') {
            return back()->with('error', 'Only ongoing deliveries can be ended.');
        }

        $delivery->update([
            'status' => 'unloading',
            'ended_at' => now(),
        ]);

        return redirect()->route('admin.deliveries.unload-form', $delivery);
    }

    public function unloadForm(Delivery $delivery)
    {
        if ($delivery->status !== 'unloading') {
            return redirect()->route('admin.deliveries.index')->with('error', 'Delivery is not in unloading status.');
        }

        $delivery->load(['items.product', 'vehicle']);
        
        // Fetch current vehicle stock for these products
        foreach ($delivery->items as $item) {
            $vs = VehicleStock::where('vehicle_id', $delivery->vehicle_id)
                ->where('product_id', $item->product_id)
                ->first();
            $item->system_quantity = $vs ? $vs->quantity : 0;
        }

        return view('admin.deliveries.unload', compact('delivery'));
    }

    public function unload(Request $request, Delivery $delivery)
    {
        $request->validate([
            'physical_quantities' => 'required|array',
            'physical_quantities.*.*' => 'required|integer|min:0',
        ]);

        try {
            DB::transaction(function () use ($request, $delivery) {
                foreach ($request->physical_quantities as $itemId => $data) {
                    $item = DeliveryItem::find($itemId);
                    $physicalQty = $data['quantity'];
                    
                    // 1. Get current vehicle stock (system quantity)
                    $vehicleStock = VehicleStock::where('vehicle_id', $delivery->vehicle_id)
                        ->where('product_id', $item->product_id)
                        ->first();
                    
                    $systemQty = $vehicleStock ? $vehicleStock->quantity : 0;

                    // 2. Record in DeliveryItem
                    $item->update([
                        'unloaded_quantity' => $systemQty,
                        'physical_unloaded_quantity' => $physicalQty,
                    ]);

                    // 3. Move physical goods back to warehouse
                    $warehouseStock = Stock::firstOrCreate(
                        ['product_id' => $item->product_id],
                        ['quantity' => 0]
                    );
                    $warehouseStock->increment('quantity', $physicalQty);

                    // 4. Reset vehicle stock
                    if ($vehicleStock) {
                        $vehicleStock->update(['quantity' => 0]);
                    }
                }

                // 5. Complete Delivery
                $delivery->update(['status' => 'completed']);
                
                // 6. Unassign Vehicle from Rep
                $rep = User::find($delivery->sales_rep_id);
                $rep->vehicle_id = null;
                $rep->save();
            });
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }

        return redirect()->route('admin.deliveries.index')->with('success', 'Truck unloaded and delivery completed!');
    }
    
    public function show(Delivery $delivery)
    {
        $delivery->load(['items.product', 'vehicle', 'driver', 'salesRep']);
        return view('admin.deliveries.show', compact('delivery'));
    }
}
