<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use App\Models\Vehicle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

class SalesRepController extends Controller
{
    public function index()
    {
        $reps = User::with('vehicle')->whereHas('role', function($q) {
            $q->where('name', 'Sales Representative');
        })->latest()->paginate(10);
        
        return view('admin.reps.index', compact('reps'));
    }

    public function create()
    {
        // Get vehicles that are NOT assigned to any rep? Or allow reassignment.
        // For simple logic, show all.
        $vehicles = Vehicle::where('is_active', true)->get();
        return view('admin.reps.create', compact('vehicles'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'mobile' => ['required', 'string', 'max:20', 'unique:'.User::class],
            'address' => ['nullable', 'string', 'max:500'],
            'vehicle_id' => ['nullable', 'exists:vehicles,id'],
            'commission_cash' => ['required', 'numeric', 'min:0', 'max:100'],
            'commission_credit' => ['required', 'numeric', 'min:0', 'max:100'],
            'commission_cheque' => ['required', 'numeric', 'min:0', 'max:100'],
        ]);

        $repRole = Role::where('name', 'Sales Representative')->firstOrFail();

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role_id' => $repRole->id,
            'mobile' => $request->mobile,
            'address' => $request->address,
            'vehicle_id' => $request->vehicle_id,
            'commission_cash' => $request->commission_cash,
            'commission_credit' => $request->commission_credit,
            'commission_cheque' => $request->commission_cheque,
            'is_active' => true,
        ]);

        return redirect()->route('admin.reps.index')->with('success', 'Sales Representative Created Successfully!');
    }
    public function edit(User $rep)
    {
        // $rep parameter name matches route resource convention 'reps/{rep}/edit'
        if (!$rep->isSalesRep()) {
            abort(404);
        }
        
        $vehicles = Vehicle::where('is_active', true)->get();
        return view('admin.reps.edit', compact('rep', 'vehicles'));
    }

    public function update(Request $request, User $rep)
    {
        if (!$rep->isSalesRep()) {
            abort(404);
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:users,email,'.$rep->id],
            'mobile' => ['required', 'string', 'max:20', 'unique:users,mobile,'.$rep->id],
            'address' => ['nullable', 'string', 'max:500'],
            'vehicle_id' => ['nullable', 'exists:vehicles,id'],
            'commission_cash' => ['required', 'numeric', 'min:0', 'max:100'],
            'commission_credit' => ['required', 'numeric', 'min:0', 'max:100'],
            'commission_cheque' => ['required', 'numeric', 'min:0', 'max:100'],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
        ]);

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'mobile' => $request->mobile,
            'address' => $request->address,
            'vehicle_id' => $request->vehicle_id,
            'commission_cash' => $request->commission_cash,
            'commission_credit' => $request->commission_credit,
            'commission_cheque' => $request->commission_cheque,
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $rep->update($data);

        return redirect()->route('admin.reps.index')->with('success', 'Sales Representative Updated Successfully!');
    }
}
