<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use Illuminate\Http\Request;
use PDF;

class PublicReceiptController extends Controller
{
    public function show(Request $request, $token)
    {
        // Find all payments with this token
        $payments = Payment::where('public_token', $token)
            ->with(['sale', 'customer', 'collector'])
            ->get();

        if ($payments->isEmpty()) {
            abort(404, 'Receipt not found.');
        }

        $customer = $payments->first()->customer;
        // Load ALL sales for total outstanding calc
        $customer->load('sales');
        $customer->current_outstanding = $customer->sales->sum('balance_due');
        
        $collector = $payments->first()->collector;
        $totalPaid = $payments->sum('amount');
        
        // Paid Allocations
        $allocations = $payments->map(function($p) {
            return [
                'invoice' => $p->sale->invoice_number,
                'amount' => $p->amount,
                'sale_balance_after' => $p->sale->balance_due
            ];
        });

        // Other Unpaid Invoices
        $otherUnpaid = $customer->sales->where('balance_due', '>', 0.01);

        $data = compact('payments', 'customer', 'collector', 'totalPaid', 'allocations', 'otherUnpaid');

        // Check if download requested
        if ($request->has('download') && $request->download == 'pdf') {
            $pdf = PDF::loadView('sales.payments.receipt_pdf', $data);
            return $pdf->download('receipt_' . $token . '.pdf');
        }

        // Return view reusing the Rep's receipt view 
        // We might want to ensure the view doesn't have sensitive Rep nav links? 
        // The layouts.app usually includes nav. 
        // Ideally we should use a public layout or just the PDF view content styled for web?
        // Let's use the same view but maybe hide nav with CSS or check auth.
        // However, standard view extends 'layouts.app'.
        // For public access, it's better to force a simple layout or just use the PDF view rendered as HTML if simple.
        // Or render the PDF view but return as view?
        // The user asked "anyone with link can view". Best to use a simple "guest" layout if possible.
        // For now, let's use the 'sales.payments.receipt' but we need to supply 'collector' etc.
        // The view extends <x-app-layout>. If user is not logged in, it might redirect to login or show generic.
        // If we want public access without login, we cannot extend 'layouts.app' if it enforces auth check content.
        // Actually layouts.app is usually just a wrapper.
        // Let's create a simplified view wrapper if needed? 
        // Or just return the existing view and ensure the layout handles guests gracefully?
        // Let's try to return the existing view. If it fails due to Auth::user calls in layout, we'll fix.
        // The layout 'navigation.blade.php' uses Auth::user(). So guest will crash it.
        // We will make a simple public layout or use a public-compatible view.
        
        // Strategy: Use a clone of the receipt view but extending 'layouts.guest' or standalone HTML.
        // Let's assume we can use a 'guest' layout.
        
        return view('sales.payments.receipt_public', $data);
    }
}
