<?php

namespace App\Http\Controllers\Sales;

use App\Http\Controllers\Controller;
use App\Models\Commission;
use App\Models\Customer;
use App\Models\Payment;
use App\Models\Product;
use App\Models\ProductPackSize;
use App\Models\ProductPriceLevel;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\VehicleStock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use PDF;

class BillingController extends Controller
{
    public function index()
    {
        // Check assigned vehicle
        $user = Auth::user();
        if (!$user->vehicle_id) {
            return view('sales.stock.no_vehicle');
        }

        $customers = Customer::where('sales_rep_id', $user->id)
            ->with(['sales'])
            ->get();
        // Products available IN TRUCK stock only
        $products = Product::where('is_active', true)
            ->whereHas('vehicleStocks', function($query) use ($user) {
                $query->where('vehicle_id', $user->vehicle_id)
                      ->where('quantity', '>', 0);
            })
            ->with(['packSizes', 'packSizes.priceLevels', 'vehicleStocks' => function($query) use ($user) {
                $query->where('vehicle_id', $user->vehicle_id);
            }])
            ->get();
        
        // Append current stock for easy frontend use
        $products->each(function($product) {
            $product->current_stock = $product->vehicleStocks->first()->quantity ?? 0;
        });
        
        return view('sales.billing.index', compact('customers', 'products', 'user'));
    }

    public function createCustomer()
    {
        return view('sales.customers.create');
    }

    public function storeCustomer(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'mobile' => 'required|string|max:20|unique:customers,mobile',
            'address' => 'required|string|max:500',
            'area' => 'nullable|string|max:255',
            'gst_number' => 'nullable|string|max:50',
            'default_price_level' => 'required|in:VVIP,VIP,Wholesale,Retail',
        ]);

        $customer = Customer::create([
            'name' => $request->name,
            'mobile' => $request->mobile,
            'address' => $request->address,
            'area' => $request->area,
            'gst_number' => $request->gst_number,
            'default_price_level' => $request->default_price_level,
            'sales_rep_id' => Auth::id(),
            'is_active' => true,
        ]);

        return redirect()->route('sales.billing')->with('success', 'Customer created successfully!');
    }

    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'items' => 'required|array',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.product_pack_size_id' => 'required|exists:product_pack_sizes,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0', // Manually entered or selected
            'items.*.price_level' => 'required|string', // VVIP, VIP, etc.
            
            // Payment inputs are optional (cash/cheque), logic handles the rest
            'payment_cash' => 'nullable|numeric|min:0',
            'payment_cheque' => 'nullable|numeric|min:0',
        ]);

        $saleId = null;

        try {
            DB::transaction(function () use ($request, &$saleId) {
                $rep = Auth::user();
                $vehicleId = $rep->vehicle_id; // Auto-assigned
                
                if (!$vehicleId) {
                    throw new \Exception("No vehicle assigned.");
                }

                // 1. Calculate Totals and Validate Stock
                $subTotal = 0;
                foreach ($request->items as $item) {
                    $subTotal += $item['quantity'] * $item['unit_price'];
                    
                    // Stock Check
                    $packSize = ProductPackSize::find($item['product_pack_size_id']);
                    $totalBase = $item['quantity'] * $packSize->quantity;
                    
                    $vehicleStock = VehicleStock::where('vehicle_id', $vehicleId)
                                        ->where('product_id', $item['product_id'])
                                        ->first();

                    $available = $vehicleStock ? $vehicleStock->quantity : 0;

                    if ($available < $totalBase) {
                        throw new \Exception("Insufficient stock for Product ID: " . $item['product_id'] . ". Available: {$available}, Requested: {$totalBase}");
                    }
                    
                    $vehicleStock->decrement('quantity', $totalBase);
                }
                
                $netTotal = $subTotal;

                // Payment Amounts
                $cashAmount = $request->payment_cash ?? 0;
                $chequeAmount = $request->payment_cheque ?? 0;
                $totalPaid = $cashAmount + $chequeAmount;
                $balanceDue = $netTotal - $totalPaid;
                
                // Payment Status
                if ($balanceDue <= 0.01) { // Floating point tolerance
                    $paymentStatus = 'paid';
                    $balanceDue = 0; // normalize
                } elseif ($totalPaid > 0) {
                    $paymentStatus = 'partial';
                } else {
                    $paymentStatus = 'pending';
                }

                // Determine Payment Method Label
                $methods = [];
                if ($cashAmount > 0) $methods[] = 'Cash';
                if ($chequeAmount > 0) $methods[] = 'Cheque';
                if ($balanceDue > 0) $methods[] = 'Credit';
                $paymentType = implode(' + ', $methods);

                // 2. Create Sale
                $publicToken = \Illuminate\Support\Str::random(32);
                $sale = Sale::create([
                    'invoice_number' => 'INV-' . time() . '-' . $rep->id,
                    'customer_id' => $request->customer_id,
                    'sales_rep_id' => $rep->id,
                    'vehicle_id' => $vehicleId,
                    'sub_total' => $subTotal,
                    'net_total' => $netTotal,
                    'paid_amount' => $totalPaid,
                    'balance_due' => $balanceDue,
                    'payment_status' => $paymentStatus,
                    'payment_type' => $paymentType,
                    'status' => 'completed',
                    'public_token' => $publicToken,
                ]);
                $saleId = $sale->id;

                // 3. Create Sale Items
                foreach ($request->items as $item) {
                     SaleItem::create([
                        'sale_id' => $sale->id,
                        'product_id' => $item['product_id'],
                        'product_pack_size_id' => $item['product_pack_size_id'],
                        'price_level_applied' => $item['price_level'],
                        'quantity' => $item['quantity'],
                        'unit_price' => $item['unit_price'],
                        'total_price' => $item['quantity'] * $item['unit_price'],
                    ]);
                }

                // 4. Record Payments
                if ($cashAmount > 0) {
                    Payment::create([
                        'sale_id' => $sale->id,
                        'customer_id' => $request->customer_id,
                        'amount' => $cashAmount,
                        'method' => 'Cash',
                        'payment_date' => now(),
                        'collected_by' => $rep->id,
                    ]);
                }
                if ($chequeAmount > 0) {
                    Payment::create([
                        'sale_id' => $sale->id,
                        'customer_id' => $request->customer_id,
                        'amount' => $chequeAmount,
                        'method' => 'Cheque',
                        'reference' => $request->cheque_reference,
                        'payment_date' => now(),
                        'collected_by' => $rep->id,
                    ]);
                }

                // 5. Send SMS
                try {
                    $customer = \App\Models\Customer::find($request->customer_id);
                    if ($customer && $customer->mobile) {
                        $notify = new \App\Services\NotifyLKService();
                        $link = route('public.invoice.show', $publicToken);
                        $message = "View Your Invoice - {$link}";
                        $notify->sendSMS($customer->mobile, $message);
                    }
                } catch (\Exception $e) {
                    \Illuminate\Support\Facades\Log::error("SMS Sending Failed: " . $e->getMessage());
                }
            });
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage())->withInput();
        }

        // Redirect to generate PDF
        return redirect()->route('sales.billing.print', $saleId)->with('success', 'Bill Created & SMS Sent!');
    }

    public function print($id)
    {
        $sale = Sale::with(['customer', 'items.product', 'items.packSize', 'salesRep'])->findOrFail($id);
        
        $pdf = PDF::loadView('sales.billing.pdf', compact('sale'));
        
        // return $pdf->download('invoice_'.$sale->invoice_number.'.pdf');
        return $pdf->stream('invoice_'.$sale->invoice_number.'.pdf');
    }
}
