<?php

namespace App\Http\Controllers\SalesRep;

use App\Http\Controllers\Controller;
use App\Models\Payroll;
use App\Models\SalaryAdvance;
use App\Models\Loan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SalaryController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Get payroll history
        $payrolls = Payroll::where('user_id', $user->id)
            ->orderBy('month', 'desc')
            ->paginate(12);

        // Get current month payroll
        $currentMonth = now()->format('Y-m');
        $currentPayroll = Payroll::where('user_id', $user->id)
            ->where('month', $currentMonth)
            ->first();

        if (!$currentPayroll) {
            $startOfMonth = now()->startOfMonth();
            $endOfMonth = now()->endOfMonth();
            
            $totalCommission = \App\Models\Commission::where('sales_rep_id', $user->id)
                ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->where('status', 'approved')
                ->sum('commission_amount');

            $basicSalary = $user->basic_salary ?? 0;
            $grossSalary = $basicSalary + $totalCommission;
            
            // Estimated Deductions (EPF 8%, ETF 3%)
            $epfEmployee = $grossSalary * 0.08;
            $etfEmployee = $grossSalary * 0.03;
            $totalDeductions = $epfEmployee + $etfEmployee; // Simplified projection
            
            $currentPayroll = new Payroll();
            $currentPayroll->month = $currentMonth;
            $currentPayroll->status = 'projected'; // Will appear as yellow/pending
            $currentPayroll->basic_salary = $basicSalary;
            $currentPayroll->total_commission = $totalCommission;
            $currentPayroll->gross_salary = $grossSalary;
            $currentPayroll->total_deductions = $totalDeductions;
            $currentPayroll->net_salary = $grossSalary - $totalDeductions;
            // Set dummy values for others to prevent view errors if accessed
            $currentPayroll->epf_employee = $epfEmployee;
            $currentPayroll->etf_employee = $etfEmployee;
        }

        // Get salary advances
        $salaryAdvances = SalaryAdvance::where('user_id', $user->id)
            ->orderBy('advance_date', 'desc')
            ->get();

        // Get loans
        $loans = Loan::where('user_id', $user->id)
            ->orderBy('loan_date', 'desc')
            ->get();

        return view('sales_rep.salary.index', compact('payrolls', 'currentPayroll', 'salaryAdvances', 'loans', 'user'));
    }

    public function show($id)
    {
        $user = Auth::user();
        $payroll = Payroll::where('user_id', $user->id)
            ->where('id', $id)
            ->firstOrFail();

        return view('sales_rep.salary.show', compact('payroll', 'user'));
    }

    public function profile()
    {
        $user = Auth::user();
        return view('sales_rep.salary.profile', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        $request->validate([
            'bank_name' => 'nullable|string|max:255',
            'bank_account_number' => 'nullable|string|max:255',
            'bank_branch' => 'nullable|string|max:255',
        ]);

        $user = Auth::user();
        $user->update($request->only([
            'bank_name',
            'bank_account_number',
            'bank_branch'
        ]));

        return back()->with('success', 'Bank details updated successfully');
    }
}
