<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('sales', function (Blueprint $table) {
            $table->id();
            $table->string('invoice_number')->unique();
            $table->foreignId('customer_id')->constrained();
            $table->foreignId('sales_rep_id')->constrained('users');
            $table->foreignId('vehicle_id')->nullable()->constrained('vehicles');
            
            $table->decimal('sub_total', 12, 2);
            $table->decimal('discount_amount', 12, 2)->default(0);
            $table->decimal('tax_amount', 12, 2)->default(0);
            $table->decimal('net_total', 12, 2);
            
            $table->decimal('paid_amount', 12, 2)->default(0);
            $table->decimal('balance_due', 12, 2)->default(0);
            
            $table->string('payment_status')->default('pending'); // pending, partial, paid
            $table->string('payment_type')->nullable(); // Cash, Credit, Mixed
            
            $table->string('status')->default('completed'); // completed, cancelled
            
            $table->timestamps();
            $table->softDeletes();
        });

        Schema::create('sale_items', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sale_id')->constrained()->onDelete('cascade');
            $table->foreignId('product_id')->constrained();
            $table->foreignId('product_pack_size_id')->constrained('product_pack_sizes');
            
            $table->string('price_level_applied'); // e.g. retail
            $table->integer('quantity');
            $table->decimal('unit_price', 12, 2);
            $table->decimal('total_price', 12, 2);
            
            $table->timestamps();
            $table->softDeletes();
        });

        Schema::create('payments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sale_id')->nullable()->constrained(); // Can be payment for multiple sales? For now 1-1 or many-1. Let's say linked to structure. 
            // Better: Payments linked to Customer with optional Sale ID?
            // "Part payment (cash + credit)".
            // Usually payments are against an invoice. If generic payment on account, sale_id is null.
            $table->foreignId('customer_id')->constrained();
            $table->decimal('amount', 12, 2);
            $table->string('method'); // Cash, Cheque
            $table->string('reference')->nullable(); // Cheque No
            $table->date('payment_date');
            $table->foreignId('collected_by')->constrained('users'); // Rep
            $table->timestamps();
            $table->softDeletes();
        });

        Schema::create('commissions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sales_rep_id')->constrained('users');
            $table->foreignId('sale_id')->constrained();
            $table->decimal('commission_amount', 12, 2);
            $table->decimal('percentage', 5, 2)->nullable(); // Store rate used
            $table->string('status')->default('pending'); // pending, paid
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('commissions');
        Schema::dropIfExists('payments');
        Schema::dropIfExists('sale_items');
        Schema::dropIfExists('sales');
    }
};
